#ifndef AHLGREN_STATISTICS
#define AHLGREN_STATISTICS

#include <array>


namespace lp {
	class stat {
	public:
		typedef double value_type;
		enum { cl_added, time, fitc, exc, conc, inconc, complexity, exhausted, nodec, bottomc, invalid,
			aborted, full_abort,
			true_positive, false_positive, true_negative, false_negative,
			size
		};
		typedef std::array<value_type,size> array_type;
		typedef array_type::iterator iterator;
		typedef array_type::const_iterator const_iterator;
		typedef array_type::size_type size_type;

		// Array containing all values
		array_type arr;

		// Constructor
		stat() { arr.fill(value_type(0)); }
		stat(const stat& s) { arr = s.arr; }
		stat& operator=(const stat& s) { if (this != &s) { arr = s.arr; } return *this; }

		// Access array
		value_type& operator[](int i) { return arr[i]; }
		value_type operator[](int i) const { return arr[i]; }

		// Iterators
		iterator begin() { return arr.begin(); }
		iterator end() { return arr.end(); }
		const_iterator begin() const { return arr.begin(); }
		const_iterator end() const { return arr.end(); }
	protected:
	};

	template <typename Binop>
	auto binop(const stat& s1, const stat& s2, Binop fun) -> stat
	{
		stat res;
		std::transform(s1.begin(),s1.end(),s2.begin(),res.begin(),fun);
		return res;
	}

	inline auto operator+(const stat& s1, const stat& s2) -> stat
	{
		return binop(s1,s2,[](double x, double y){ return x+y; });
	}

	// template <typename double, typename double>
	inline auto operator-(const stat& s1, const stat& s2) -> stat
	{
		return binop(s1,s2,[](double x, double y){ return x-y; });
	}

	// template <typename double, typename double>
	inline auto operator*(const stat& s1, const stat& s2) -> stat
	{
		return binop(s1,s2,[](double x, double y){ return x*y; });
	}

	// template <typename double, typename double>
	inline auto min(const stat& s1, const stat& s2) -> stat
	{
		return binop(s1,s2,[](double x, double y){ return x < y ? x : y; });
	}

	// template <typename double, typename double>
	inline auto max(const stat& s1, const stat& s2) -> stat
	{
		return binop(s1,s2,[](double x, double y){ return x > y ? x : y; });
	}

	inline stat operator*(double d, const stat& s)
	{
		stat res;
		std::transform(s.begin(),s.end(),res.begin(),[d](double x){ return d * x; });
		return res;
	}

	inline stat operator/(const stat& s, double d)
	{
		stat res;
		std::transform(s.begin(),s.end(),res.begin(),[d](double x){ return x / d; });
		return res;
	}

	inline stat sqrt(const stat& s)
	{
		stat res;
		std::transform(s.begin(),s.end(),res.begin(),[](double x){ return std::sqrt(x); });
		return res;
	}

	template <typename Iter>
	auto sum(Iter beg, Iter end) -> typename std::iterator_traits<Iter>::value_type
	{
		return std::accumulate(beg,end,typename std::iterator_traits<Iter>::value_type());
	}

	template <typename Iter>
	auto sqsum(Iter beg, Iter end) -> typename std::iterator_traits<Iter>::value_type
	{
		typedef typename std::iterator_traits<Iter>::value_type stat_type;
		return std::inner_product(beg,end,beg,stat_type());
		//return std::accumulate(beg,end,stat_type(),
		//	[](const stat_type& s1, const stat_type& s2){ return s1+s2*s2; });
	}

	template <typename Iter>
	stat average(Iter beg, Iter end)
	{
		return sum(beg,end) / double(std::distance(beg,end));
	}

	template <typename Iter>
	stat variance(Iter beg, Iter end)
	{
		const stat avg = average(beg,end);
		const double n = std::distance(beg,end);
		return (sqsum(beg,end) - n*avg*avg) / (n-1);
	}

	template <typename Iter>
	stat stddev(Iter beg, Iter end)
	{
		return sqrt(variance(beg,end));
	}


	template <typename Iter>
	auto stat_min(Iter beg, Iter end) -> typename std::iterator_traits<Iter>::value_type
	{
		typedef typename std::iterator_traits<Iter>::value_type stat_type;
		auto first = beg;
		return std::accumulate(++beg,end,*first,[](const stat_type& s, const stat_type& t){ return min(s,t); });
	}

	template <typename Iter>
	auto stat_max(Iter beg, Iter end) -> typename std::iterator_traits<Iter>::value_type
	{
		typedef typename std::iterator_traits<Iter>::value_type stat_type;
		auto first = beg;
		return std::accumulate(++beg,end,*first,[](const stat_type& s, const stat_type& t){ return max(s,t); });
	}

	//inline ostream& operator<<(ostream& os, const statistics& stats) 
	//{
	//	const stat sum = stats.sum();

	//	const stat var = stats.square_sum() - sum*sum;
	//	os << "Generate Statistics:\n";
	//	os << "Sample size:                " << stats.samples() << "\n";
	//	os << "Accumulated Statistics:\n";
	//	os << "Clauses Added:              " << sum.cl_added << "\n";
	//	os << "Total Time:                 " << sum.time << "\n";
	//	os << "Total Fitness Calls:        " << sum.fitc << "\n";
	//	os << "Total Examples Evaluated:   " << sum.exc << "\n";
	//	os << "Consistent Candidates:      " << sum.consistentc << "\n";
	//	os << "Inconsistent Candidates:    " << sum.inconsistentc << "\n";
	//	os << "\n";
	//	const double samples = stats.samples();
	//	os << "Average Statistics:\n";
	//	os << "Avg Clauses Added:          " << sum.cl_added / samples << "\n";
	//	os << "Avg Time:                   " << sum.time / samples << "\n";
	//	os << "Avg Fitness Calls:          " << sum.fitc / samples << "\n";
	//	os << "Avg Examples Evaluated:     " << sum.exc / samples << "\n";
	//	os << "Percentage Consistent:      " << 100.0 * sum.consistentc / (sum.consistentc+sum.inconsistentc) << "\n";
	//	os << "Percentage Complete:        " << double(sum.complete) / samples << "\n";
	//	os << "Avg Complexity:             " << sum.complexity / samples << "\n";
	//	os << "\n";
	//	return os;
	//}
	
} // namespace lp

#endif /* AHLGREN_STATISTICS */
